<?

namespace ExportEngine\Drivers;

use League\Csv\Writer;

/**
 * Description of Csv
 *
 * @author d.lanec
 */
class Csv implements \ExportEngine\Contracts\ExportProcessInterface
{
	
	protected $file;
	protected $data;
	protected $separator = ";";
	protected $enclosure = '"';
    protected $escape = '\\';
	protected $linebreak = "\r\n";
	protected $fileName;
	protected $charSet;
	protected $head;

	static $ext = "csv";
	
	function __construct(\SplFileObject $file)
	{
		$this->file = $file;
	}
	
	/**
	 * 
	 * @param array $head
	 * @return $this
	 */
	public function setHead(array $head)
	{
		$this->head = $head;
		return $this;
	}
	
	/**
	 * 
	 * @param type $charSet
	 * @return $this
	 */
	public function setCharSet($charSet)
	{
		$this->charSet = $charSet;
		return $this;
	}
	
	/**
	 * 
	 * @return string
	 */
	public function getExtendsion()
	{
		return self::$ext;
	}
	
	/**
	 * 
	 * @param type $name
	 * @return $this
	 */
	public function setFileName($name){
		$this->fileName = $name;
		return $this;
	}

	/**
	 * 
	 * @param type $separator
	 * @return $this
	 */
	public function setSeparator($separator)
	{
		$this->separator = $separator;
		return $this;
	}
	
    /**
     * @param $enclosure
     * @return $this
     */
    public function setEnclosure($enclosure)
    {
        $this->enclosure = $enclosure;
        return $this;
    }

    /**
     * @param $escape
     * @return $this
     */
    public function setEscape($escape)
    {
        $this->escape = $escape;
        return $this;
    }

	/**
	 * 
	 * @param type $linebreak
	 * @return $this
	 */
	public function setLinebreak($linebreak)
	{
		$this->linebreak = $linebreak;
		return $this;
	}

	/**
	 * 
	 * @return string
	 */
	public function output()
	{

		$filename = filter_var($this->fileName, FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
		header('Content-Transfer-Encoding: binary');
		header("Content-Disposition: attachment; filename=\"$filename\"");
		header("Content-Type: text/csv; charset=" . $this->charSet);
        return $this->getContentString();
    }
	
	/**
	 * save file to path
	 * @param type $path
	 */
	public function exportToFile($path, $flag = 0)
    {
        if(!$content = $this->getContentString()){
            return;
        }

        if(!$this->fp){
            $this->fp = fopen($path,"w+");

            @flock($this->fp, LOCK_SH);
        }

        if(fwrite($this->fp, $content) === false ){
            throw new \Exception("write file error ". $path);
        }
    }

		/**
	 * 
	 * @param array $data
	 * @return $this
	 */
	public function setData(array $data)
	{
		$this->data = $data;
		return $this;
	}

    /**
     * @return string
     */
    protected function getContentString()
    {
        $content = $this->createWriter()->__toString();
        if ($this->charSet == 'windows-1251') {
            $content = iconv("utf-8", $this->charSet . '//TRANSLIT//IGNORE', $content);
        }
        return $content;
    }
	
	/**
	 * 
	 * @return \League\Csv\Writer
	 */
	protected function createWriter(){
		
        $writer = Writer::createFromStream(tmpfile());

		$writer->setDelimiter($this->separator);
		$writer->setEnclosure($this->enclosure);
		$writer->setEscape($this->escape);
		$writer->setNewline($this->linebreak);

        if ($this->charSet == 'utf-8') {
            $writer->setOutputBOM(Writer::BOM_UTF8); //adding the BOM sequence on output
        }

        if (!empty($this->head)) {
            $writer->insertOne($this->head);
        }

		$writer->insertAll($this->data);

		return $writer;	
	}


	public function __destruct()
    {

        if($this->fp){
            flock($this->fp, LOCK_UN);
            fclose($this->fp);
        }

    }

}