<?php

namespace ExportEngine\Concrete\Drivers;

use ExportEngine\Contracts\DriverInterface;
use ExportEngine\Contracts\FileLinesConfiguratorInterface;
use ExportEngine\Exceptions\ExportException;
use ExportEngine\Storage\FileFormat;
use ExportEngine\Traits\SpreadsheetAwareTrait;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Csv as CsvWriter;
use PhpOffice\PhpSpreadsheet\Writer\Exception as PhpSpreadsheetException;

/**
 * Class TxtDriver
 *
 * @author Aleksei Kuznetsov
 */
class TxtDriver implements DriverInterface
{
    use SpreadsheetAwareTrait;

    /**
     * @var FileLinesConfiguratorInterface
     */
    protected $fileConfig;

    /**
     * HtmlDriver constructor.
     *
     * @param FileLinesConfiguratorInterface $configuration
     */
    public function __construct(FileLinesConfiguratorInterface $configuration)
    {
        $this->fileConfig = $configuration;
        $this->fileConfig->setExtension(FileFormat::TXT);

        $this->spreadsheet = new Spreadsheet();
    }

    /**
     * @return bool
     * @throws ExportException
     */
    public function export(): bool
    {
        $writer = $this->getWriter();

        try {
            $writer->save($this->fileConfig->getFilePath());
        } catch (PhpSpreadsheetException $e) {
            throw new ExportException($e->getMessage());
        }

        $this->clearData();

        return true;
    }

    /**
     * @return CsvWriter
     */
    private function getWriter(): CsvWriter
    {
        $writer = new CsvWriter($this->spreadsheet);

        $writer->setDelimiter($this->fileConfig->getDelimiter());
        $writer->setEnclosure($this->fileConfig->getEnclosure());
        $writer->setLineEnding($this->fileConfig->getLineEnding());

        if ($this->fileConfig->getCharset() === 'UTF-8') {
            $writer->setUseBOM(true);
        }

        return $writer;
    }

    /**
     * @throws ExportException
     */
    public function showOnScreen(): void
    {
        $this->export();

        $this->sendHTTPHeaders();
        readfile($this->fileConfig->getFilePath());

        exit;
    }

    /**
     *
     */
    private function sendHTTPHeaders(): void
    {
        header('Content-Description: File Transfer');
        header('Content-Transfer-Encoding: binary');
        header('Pragma: no-cache');
        header('Expires: 0');
        header(
            'Content-Disposition: attachment; filename="'
            . filter_var($this->fileConfig->getFileName(), FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW)
            . '"'
        );
        header('Content-Length: ' . filesize($this->fileConfig->getFilePath()));
        header('Content-Type: text/csv; charset=' . $this->fileConfig->getCharset());
    }
}
