<?php

namespace ExportEngine\Factories;

use ExportEngine\Concrete\Drivers\CsvDriver;
use ExportEngine\Concrete\Drivers\HtmlDriver;
use ExportEngine\Concrete\Drivers\TxtDriver;
use ExportEngine\Concrete\Drivers\XlsDriver;
use ExportEngine\Concrete\Drivers\XlsxDriver;
use ExportEngine\Concrete\Drivers\XmlDriver;
use ExportEngine\Contracts\DriverInterface;
use ExportEngine\Contracts\FileConfiguratorInterface;
use ExportEngine\Contracts\FileLinesConfiguratorInterface;
use ExportEngine\Exceptions\FileFormatNotSupportedException;

/**
 * Фабрика драйверов
 *
 * @author d.lanec
 */
class DriverFactory
{
    /**
     * Создание объекта по типу
     *
     * @param string $fileType
     * @param FileConfiguratorInterface|FileLinesConfiguratorInterface $fileConfig
     *
     * @return DriverInterface
     * @throws FileFormatNotSupportedException
     */
    public static function buildByFileType(string $fileType, FileConfiguratorInterface $fileConfig): DriverInterface
    {
        $methodName = 'build' . ucfirst(strtolower($fileType));

        if (!method_exists(self::class, $methodName)) {
            throw new FileFormatNotSupportedException($fileType);
        }

        return self::$methodName($fileConfig);
    }

    /**
     * @param FileLinesConfiguratorInterface $fileConfig
     *
     * @return CsvDriver
     */
    public static function buildCsv(FileLinesConfiguratorInterface $fileConfig): CsvDriver
    {
        return new CsvDriver($fileConfig);
    }

    /**
     * @param FileConfiguratorInterface $fileConfig
     *
     * @return HtmlDriver
     */
    public static function buildHtml(FileConfiguratorInterface $fileConfig): HtmlDriver
    {
        return new HtmlDriver($fileConfig);
    }

    /**
     * @param FileLinesConfiguratorInterface $fileConfig
     *
     * @return TxtDriver
     */
    public static function buildTxt(FileLinesConfiguratorInterface $fileConfig): TxtDriver
    {
        return new TxtDriver($fileConfig);
    }

    /**
     * @param FileConfiguratorInterface $fileConfig
     *
     * @return XlsDriver
     */
    public static function buildXls(FileConfiguratorInterface $fileConfig): XlsDriver
    {
        return new XlsDriver($fileConfig);
    }

    /**
     * @param FileConfiguratorInterface $fileConfig
     *
     * @return XlsxDriver
     */
    public static function buildXlsx(FileConfiguratorInterface $fileConfig): XlsxDriver
    {
        return new XlsxDriver($fileConfig);
    }

    /**
     * @param FileConfiguratorInterface $fileConfig
     *
     * @return XmlDriver
     */
    public static function buildXml(FileConfiguratorInterface $fileConfig): XmlDriver
    {
        return new XmlDriver($fileConfig);
    }
}
