<?php

namespace ExportEngine\Concrete\Drivers;

use ExportEngine\Contracts\DriverInterface;
use ExportEngine\Contracts\FileConfiguratorInterface;
use ExportEngine\Exceptions\ExportException;
use ExportEngine\Helpers\XmlWriter;
use ExportEngine\Storage\FileFormat;
use ExportEngine\Traits\SpreadsheetAwareTrait;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Exception as PhpSpreadsheetException;

/**
 * Class XmlDriver
 *
 * @author Aleksei Kuznetsov
 */
class XmlDriver implements DriverInterface
{
    use SpreadsheetAwareTrait;

    /**
     * @var FileConfiguratorInterface
     */
    protected $fileConfig;
    
    /**
     * XmlDriver constructor.
     *
     * @param FileConfiguratorInterface $configuration
     */
    public function __construct(FileConfiguratorInterface $configuration)
    {
        $this->fileConfig = $configuration;
        $this->fileConfig->setExtension(FileFormat::XML);

        $this->spreadsheet = new Spreadsheet();
    }

    /**
     * @return bool
     * @throws ExportException
     */
    public function export(): bool
    {
        $writer = $this->getWriter();

        try {
            $writer->save($this->fileConfig->getFilePath());
        } catch (PhpSpreadsheetException $e) {
            throw new ExportException($e->getMessage());
        }

        $this->clearData();

        return true;
    }

    /**
     * @return XmlWriter
     */
    private function getWriter(): XmlWriter
    {
        return new XmlWriter($this->spreadsheet);
    }

    /**
     * @throws ExportException
     */
    public function showOnScreen(): void
    {
        $this->export();

        $this->sendHTTPHeaders();
        readfile($this->fileConfig->getFilePath());

        exit;
    }

    /**
     *
     */
    private function sendHTTPHeaders(): void
    {
        header('Content-Description: File Transfer');
        header('Content-Transfer-Encoding: binary');
        header('Pragma: no-cache');
        header('Expires: 0');
        header(
            'Content-Disposition: attachment; filename="'
            . filter_var($this->fileConfig->getFileName(), FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW)
            . '"'
        );
        header('Content-Length: ' . filesize($this->fileConfig->getFilePath()));
        header('Content-Type: application/xml;');
    }
}
