<?php

namespace Core\Form;

use Core\Form\Control\Checkbox;
use Core\Form\Control\DatePickerRange;
use Core\Form\Control\Input;
use Core\Form\Control\ProgressBar;
use Core\Form\Control\Switchery;
use Core\Form\Control\TableLabel;
use Core\Form\Control\Uniform\Dropdown;
use Core\Form\Control\UploadButton;

class NiftyControlBuilder extends ControlBuilder
{

    public function buildInputTag(string $name): Input
    {
        $this->template->addStyleLink('/_diamond/diamond-cms/assets/nifty/plugins/bootstrap-tagsinput/bootstrap-tagsinput.min.css');
        $this->template->addScriptLink('/_diamond/diamond-cms/assets/nifty/plugins/bootstrap-tagsinput/bootstrap-tagsinput.min.js');
        return (new Input($name))->addAttr('data-role', 'tagsinput');
    }

    /**
     * Выпадающий список с мультивыбором
     */
    public function buildNiftyMultiSelect(string $name, array $options, string $placeholder = 'Выберите из списка...'): Control\DropdownMulti
    {
        $id = 'nifty-multiselect-' . $name;
        $this->template->addScript($this->makeOnReadyFunction(
            "jQuery('#{$id}').chosen({width:'100%'});")
        );
        $this->template->addScriptLink('/_diamond/diamond-cms/assets/nifty/plugins/chosen/chosen.jquery.min.js');
        $this->template->addStyleLink('/_diamond/diamond-cms/assets/nifty/plugins/chosen/chosen.min.css');
        return (new Control\DropdownMulti($name, $options))
            ->setid($id)
            ->addAttr('data-placeholder', $placeholder);
    }

    /**
     * Выпадающий список с поиском
     */
    public function buildNiftyDropdownChosen(string $name, array $options, string $placeholder = 'Choose a value...'): Dropdown
    {
        $this->template->addScriptLink('/_diamond/diamond-cms/assets/nifty/plugins/chosen/chosen.jquery.min.js');
        $this->template->addStyleLink('/_diamond/diamond-cms/assets/nifty/plugins/chosen/chosen.min.css');
        $this->template->addScript($this->makeOnReadyFunction(
            "
            jQuery('select[name={$name}]').chosen({width:'100%'});
            "));
        return (new Dropdown($name, $options))->addData('placeholder', $placeholder);
    }

    public function buildNiftySwitchery(array $data, bool $checked): Switchery
    {
        return (new Switchery($data, $checked));
    }

    public function buildNiftySimpleSwitchery(string $name, string $checkValue, string $noCheckValue, string $color = null): Checkbox
    {
        $control = (new Checkbox($name, $checkValue, false, $noCheckValue))
            ->setSingle(false)
            ->addData('switchery', 'true')
            ->setClass('sw-checked');

        if ($color) {
            $control->addData('color', $color);
        }

        return $control;
    }


    private function makeOnReadyFunction(string $script): string
    {
        return '<script type="text/javascript">jQuery(window).load(function () { ' . $script . ' });</script>';
    }


    public function buildDatePicker(string $name): Control\DatePicker
    {
        $this->template->addScriptLink('/_diamond/diamond-cms/assets/nifty/plugins/bootstrap-datepicker/bootstrap-datepicker.min.js');
        $this->template->addStyleLink('/_diamond/diamond-cms/assets/nifty/plugins/bootstrap-datepicker/bootstrap-datepicker.min.css');
        $id = 'nifty-datepicker-' . $name;
        $this->template->addScript($this->makeOnReadyFunction(
            "jQuery('#{$id}').datepicker({
             format: \"dd.mm.yyyy\",
             todayBtn: \"linked\",
             autoclose: true,
             todayHighlight: true
             });")
        );

        return parent::buildDatePicker($name)->addClass('form-control')->setId($id);
    }

    public function buildDatePickerRange(string $nameFrom, string $nameTo): DatePickerRange
    {
        $this->template->addScriptLink('/_diamond/diamond-cms/assets/nifty/plugins/bootstrap-datepicker/bootstrap-datepicker.min.js');
        $this->template->addStyleLink('/_diamond/diamond-cms/assets/nifty/plugins/bootstrap-datepicker/bootstrap-datepicker.min.css');
        $id = strtolower($nameFrom . '_' . $nameTo);

        $control = new DatePickerRange($nameFrom, $nameTo);
        $control->setId($id);
        $this->template->addScript($this->makeOnReadyFunction(
            "jQuery('#" . $id . " .input-daterange').datepicker({
                    format: \"dd.mm.yyyy\",
                    todayBtn: \"linked\",
                    autoclose: true,
                    todayHighlight: true
                  });"
        ));

        return $control;
    }

    public function buildTableLabel(string $label, string $class): TableLabel
    {
        return new TableLabel($label, $class);
    }

    public function buildProgressBar(float $progress, string $class = 'success', bool $dinamicColor = false): ProgressBar
    {
        if ($dinamicColor === true) {
            if ($progress < 30) {
                $class = 'danger';
            } elseif ($progress < 60) {
                $class = 'warning';
            } elseif ($progress < 90) {
                $class = 'info';
            }
        }
        return new ProgressBar($progress, $class);
    }

    public function buildFileUpload(string $name, string $label): UploadButton
    {
        return new UploadButton($name, $label);
    }

}