<?php

namespace Core\Form;

use Core\Form\Constraints\Captcha;
use Core\Form\Constraints\Custom;
use Core\Form\Constraints\Email;
use Core\Form\Constraints\ReCaptcha;
use Core\Form\Control\Button;
use Core\Form\Control\ControlRenderInterface;
use Core\Form\Control\FormControl;
use Core\Form\Control\Hidden;
use Core\Form\Control\Row;
use Core\Helpers\Text;
use Core\TwigAwareInterface;
use Diamond\TwigExtensions;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Constraints;
use Symfony\Component\Validator\Validation;
use Twig_Environment;

/**
 * Простая форма
 *
 *
 * @package System
 * @subpackage library
 * @author Дмитрий
 */
class Simpleform implements TwigAwareInterface
{

    //имя формы
    protected $name;
    //объект контролов
    protected $controls = [
    ];
    //адрес формы
    protected $action;
    //поля
    protected $fields = array();
    //поля управления
    protected $fields_control = array();
    //служебные скрытые поля
    protected $hidden_fields = array();
    //на сабмит формы
    protected $onsubmit;
    protected $captions = [
    ];
    protected $validation_rules = [
    ];
    protected $fieldsdesc = [
    ];
    protected $custom_messages = [
    ];
    //действие
    //create - создание записи
    //read - чтение записи 
    //update - обновление записи 
    //delete - удаление записи
    //view - отсутсвие какого либо действия или просто просмотр формы
    private $fn;
    //строки для хранения произвольной информации
    //protected $other_rows = array();
    //Основной массив с контроллами формы
    // protected $form_fields = array();

    protected $class;
    protected $attributes = [];

    protected $classError = self::MESSAGE_ERROR;
    protected $helpText = '<small class="help-block">%s</small>';

    protected $message_blank = '<div class="alert alert-%s">%s</div>';
    //Использовать метку * при выводе обязательных полей
    protected $required_label = '<span class="required_sign">*</span>';
    protected $formRow = '<div class="form-group %s">{{%s.label}}<div class="col-md-9">{{%s.control}}</div></div>';
    protected $formHeader = '<div class="stdformrow"><span class="hd">{{%s.control}}</span></div>';
    protected $formLabel = '<label class="col-md-3 control-label" for="%s">%s</label>';
    protected $formControl = '<div class="stdformrow">{{%s.control}}</div>';
    protected $formButton = "<div class=\"stdformbutton\">%s</div>";
    protected $alertLink = 'alert-link';


    //protected $formRow = '<div class="form-group %s">{{%s.label}}<div class="col-md-9">{{%s.control}}</div></div>';
    //protected $formHeader = '<div class="panel-heading"><h3 class="panel-title">{{%s.control}}</h3></div>';
    //protected $formLabel = '<label class="col-md-3 control-label" for="%s">%s</label>';
    //protected $formControl = '<p class="stdformrow">{{%s.control}}</p>';
    //protected $formButton = "<p class=\"text-right\">%s</p>";
    //protected $classError = self::MESSAGE_ERROR;
    //protected $helpText = '<small class="help-block">%s</small>';


    protected $custom_tpl;
    //protected $hide_validation = FALSE;

    protected $formHtmlTop;
    protected $formHtmlBootom;

    /**
     * сбмит формы
     * @var type
     */
    static $ACTION_SUBMIT = 'submit';

    /**
     * отображение формы
     * @var type
     */
    static $ACTION_VIEW = 'view';

    /**
     *
     * @var Twig_Environment
     */
    protected $twig;
    static $VALIDATION_SUCCESS = 'validation_success';
    static $VALIDATION_PREPARE = 'validation_prepare';
    protected $events = [];
    protected $validation_results = [];
    protected $request;
    protected $validation;

    const MESSAGE_SUCCESS = "success";
    const MESSAGE_INFO = "info";
    const MESSAGE_ERROR = "danger";
    const MESSAGE_BACK_URL = "Вернуться назад";
    const MESSAGE_ADD_URL = "Добавить новую запись";

    protected $backUrl;
    protected $newRecordUrl;
    protected $customValidations = [];
    protected $buffer = [];

    protected $fieldClass = null;

    protected bool $nativeValidation = true;

    use \Core\TwigAwareTrait;

    function __construct($name, $action = '', Request $request = null)
    {
        $this
            ->setName($name)
            ->setAction($action);

        $this->request = !empty($request) ? $request : Request::createFromGlobals();

        $this->events[self::$VALIDATION_PREPARE] = [];
        $this->events[self::$VALIDATION_SUCCESS] = [];

        $this->validation = Validation::createValidator();
    }

    public function setAlertLink(string $alertLink): Simpleform
    {
        $this->alertLink = $alertLink;
        return $this;
    }

    public function disableNativeValidation(): self
    {
        $this->nativeValidation = false;
        return $this;
    }

    /**
     * @param null $fieldClass
     * @return Simpleform
     */
    public function setFieldClass($fieldClass)
    {
        $this->fieldClass = $fieldClass;
        return $this;
    }

    /**
     * @param string $helpText
     * @return Simpleform
     */
    public function setHelpText(string $helpText): Simpleform
    {
        $this->helpText = $helpText;
        return $this;
    }

    /**
     *
     * @return array
     */
    function getBuffer(): array
    {
        return $this->buffer;
    }

    /**
     *
     * @param type $buffer
     * @return string
     */
    function addBuffer(string $buffer)
    {
        $this->buffer [] = $buffer;
        return $this;
    }

    public function addCustomValidation($name, $callable)
    {
        $this->customValidations[$name] = $callable;
        return $this;
    }

    function setFormRow($formRow)
    {
        $this->formRow = $formRow;
        return $this;
    }

    function setFormButton($formButton)
    {
        $this->formButton = $formButton;
        return $this;
    }

    function setFormHeader($formHeader)
    {
        $this->formHeader = $formHeader;
        return $this;
    }

    function setFormLabel($formLabel)
    {
        $this->formLabel = $formLabel;
        return $this;
    }

    function setFormControl($formControl)
    {
        $this->formControl = $formControl;
        return $this;
    }

    /**
     * @param string $custom_tpl
     * @return Simpleform
     * @deprecated
     * @see Simpleform::setCustomTpl()
     */
    function setCustom_tpl(string $custom_tpl): self
    {
        return $this->setCustomTpl($custom_tpl);
    }

    /**
     * @param string $tpl
     * @return Simpleform
     */
    public function setCustomTpl(string $tpl): self
    {
        $this->custom_tpl = $tpl;
        return $this;
    }

    public function addValidationSuccessEvent($event)
    {
        $this->events[self::$VALIDATION_SUCCESS][] = $event;
        return $this;
    }

    public function addValidationPrepareEvent($event)
    {
        $this->events[self::$VALIDATION_PREPARE][] = $event;
        return $this;
    }

    function getRequired_label()
    {
        return $this->required_label;
    }

    function setRequired_label($required_label)
    {
        $this->required_label = $required_label;
        return $this;
    }

    function getName()
    {
        return $this->name;
    }

    function getAction()
    {
        return $this->action;
    }

    function getClass()
    {
        return $this->class;
    }

    function getAttributes()
    {
        return $this->attributes;
    }

    function setName($name)
    {
        $this->name = $name;
        return $this;
    }

    function setAction($action)
    {
        $this->action = $action;
        return $this;
    }

    function setClass($class)
    {
        $this->class = $class;
        return $this;
    }

    function addClass($class)
    {
        $this->class .= ' ' . $class;
        return $this;
    }

    function setAttributes($attributes)
    {
        $this->attributes = $attributes;
        return $this;
    }

    public function addHeader(string $txt)
    {
        return $this->addControl((new Control\Header($txt))->setName(str_replace('-','_',Text::rus2translit($txt))));
    }

    public function addSubHeader(string $txt)
    {
        return $this->addControl((new Control\SubHeader($txt))->setName(str_replace('-','_',Text::replaceAndTranslite($txt))));
    }

    public function addFieldControl(Control\BaseControl $control, $caption = null, $validationrule = null, $desc = null)
    {
        if (is_a($control, Control\FormField::class)) {
            return $this->addField($control, $caption, $validationrule, $desc);
        }

        return $this->addControl($control);
    }

    public function addControl(FormControl $control)
    {
        $this->fields [] = $control;
        return $this;
    }

    /**
     * @param Control\FormField $control
     * @param string|null $caption
     * @param null $validationrule
     * @param string|null $desc
     * @return $this
     */
    public function addField(ControlRenderInterface $control, string $caption = null, $validationrule = null, string $desc = null)
    {
        $this->fields [] = $control;
        $this->captions[$control->getName()] = $caption;

        if (!empty($validationrule)) {
            $this->validation_rules[$control->getName()] = $validationrule;
        }

        if (!empty($desc)) {
            $this->fieldsdesc[$control->getName()] = $desc;
        }

        return $this;
    }

    public function findFieldByName(string $name): ?FormControl
    {
        foreach ($this->fields as $field) {
            if ($field->getName() === $name) {
                return $field;
            }
        }
        return null;
    }

    public function render()
    {
        $data = $this->getPostData();

        switch ($this->definitionAction()) {
            case self::$ACTION_SUBMIT:

                $events = (array)$this->events[self::$VALIDATION_PREPARE];

                $firstForm = $this->generateForm($data);

                foreach ($events as $event) {
                    if (!$data = call_user_func($event, $data, $this)) {

                        $modForm = $this->generateForm($data);

                        if ($firstForm === $modForm) {
                            $this->addBuffer($this->createMessage('Ошибка в подготовке данных'));
                        }

                        $this->addBuffer($modForm);

                        return implode("\r\n", $this->getBuffer());
                    }
                }
                //Процесс валидации дефолтный
                if ($this->nativeValidation === true) {

                    $this->validationProcess($data);
                }

                if (count($this->validation_results) === 0) {

                    $events = (array)$this->events[self::$VALIDATION_SUCCESS];

                    foreach ($events as $event) {

                        $result = call_user_func($event, $data, $this);

                        if (!$result) {
                            break;
                        }

                    }
                    $customMessages = $this->getCustomMessages();
                    $resultHtml = implode("\r\n", $this->getBuffer());
                    if (strpos($resultHtml, $customMessages) !== false) {
                        return $resultHtml;
                    }

                    return $customMessages . $resultHtml;
                }


                break;
        }

        return $this->generateForm($data);
    }

    /**
     *
     */
    protected function validationProcess(array $data)
    {
        $validation = $this->validation;

        foreach ($this->fields as $Field) {

            if (!isset($this->validation_rules[$Field->getName()])) {
                continue;
            }

            $rules = $this->buildValidationRules($Field);

            if (!isset($data[$Field->getName()])) {
                continue;
            }

            $violations = $validation->validate($data[$Field->getName()], $rules);

            if (0 !== count($violations)) {

                // there are errors, now you can show them
                foreach ($violations as $violation) {
                    $this->validation_results[$Field->getName()] = $violation->getMessage();
                }
            }
        }
    }

    /**
     * @param Control\BaseControl $Field
     * @return array
     */
    protected function buildValidationRules(Control\BaseControl $Field): array
    {
        $rules = $this->validation_rules[$Field->getName()];

        if (!is_array($rules)) {

            $rules = explode('|', $rules);
        }
        $correct_rules = [];

        foreach ($rules as $rule) {


            switch ($rule) {
                case 'required':

                    $rule = new Constraints\NotBlank();
                    $rule->message = 'поле не должно быть пустым.';
                    $correct_rules [] = $rule;

                    break;

                case 'email':
                case 'mail':
                    $rule = new Email();
                    $correct_rules [] = $rule;

                    break;

                case 'captcha':

                    $rule = new Captcha($Field);
                    //$rule->message    = "поле не должно быть пустым.";
                    $correct_rules [] = $rule;

                    break;

                case 'recaptcha':

                    $rule = new ReCaptcha($Field);
                    $correct_rules [] = $rule;

                    break;


                default:

                    if (isset($this->customValidations[$rule])) {

                        $rule = new Custom($this->customValidations[$rule]);
                        $correct_rules [] = $rule;
                    }

                    break;
            }
        }

        return $correct_rules;
    }

    /**
     * @return string
     */
    protected function definitionAction(): string
    {

        if (!empty($this->request->request->get('submit_' . $this->getName()))) {
            return self::$ACTION_SUBMIT;
        }

        return self::$ACTION_VIEW;
    }

    /**
     * Подготовка данных  из $_POST
     * @return array
     */
    protected function getPostData(): array
    {
        $data = [];

        /**
         * пересоздание request на случай что дополнились данные
         */
        $request = Request::createFromGlobals()->request;

        foreach ($this->fields as $Field) {
            $value = $request->get($Field->getName());

            if ($value == -1) {
                $data[$Field->getName()] = null;
            } elseif ($value !== null) {
                $data[$Field->getName()] = $value;
            }

        }

        return $data;
    }

    public function generateForm(?array $posData  = null)
    {
        $data = [];

        if (!$posData) {
            $posData = $this->request->request->all();
        }

        if(!$this->twig){
            throw new \RuntimeException('twig not exist');
        }

        $template_twig = $this->twig->createTemplate($this->createTemplate());

        foreach ($this->fields as $field) {

            $fName = $this->prepareFieldName($field->getName());

            $value = $posData[$fName] ?? null;

            if ($value !== null) {
                $field->setValue($value);
            }

            $caption = $this->captions[$field->getName()]?? null;

            if ($this->isRequeredField($field->getName())) {
                $caption .= $this->required_label;
            }

            $controlHtml = $field->render();

            if (isset($this->fieldsdesc[$field->getName()])) {
                $controlHtml .= sprintf($this->helpText, $this->fieldsdesc[$field->getName()]);
            }

            $data[$fName] = [
                'control' => $controlHtml,
                'label' => sprintf($this->formLabel, $fName, $caption)
            ];
        }

        return $template_twig->render($data);
    }

    public function prepareFieldName(string $fName): string
    {
        if (strpos($fName, '[') !== false) {
            return str_replace(['[', ']'], '', $fName) . '_';
        }
        return $fName;
    }

    /**
     * @return mixed
     */
    public function getFormHtmlTop()
    {
        return $this->formHtmlTop;
    }

    /**
     * @param mixed $formHtmlTop
     * @return Simpleform
     */
    public function setFormHtmlTop($formHtmlTop)
    {
        $this->formHtmlTop = $formHtmlTop;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getFormHtmlBootom()
    {
        return $this->formHtmlBootom;
    }

    /**
     * @param mixed $formHtmlBootom
     * @return Simpleform
     */
    public function setFormHtmlBootom($formHtmlBootom)
    {
        $this->formHtmlBootom = $formHtmlBootom;
        return $this;
    }

    /**
     * @return array
     */
    public function getFields(): array
    {
        return $this->fields;
    }


    protected function createTemplate()
    {
        //выйдем
        if (isset($this->custom_tpl))
            return $this->createForm($this->custom_tpl);

        $html = '';

        if (!empty($this->formHtmlTop)) {
            $html .= $this->formHtmlTop;
        }

        foreach ($this->fields as $Field) {

            if (is_a($Field, Row::class)) {
                $html .= sprintf(
                    $this->formControl, $Field->getName()
                );
            } elseif (is_a($Field, Control\SubHeader::class)) {
                $html .= sprintf('<p class="bord-btm pad-ver text-main text-bold">{{%s.control}}</p>', $Field->getName());
            } elseif (is_a($Field, Button::class) || is_a($Field, Hidden::class) || is_a($Field, Header::class)) {

            } else {

                if ($this->fieldClass !== null) {
                    $Field->addClass($this->fieldClass);
                }

                if ($this->isRequeredField($Field->getName())) {
                    $required_class = 'required';
                } else {
                    $required_class = '';
                }

                $html .= sprintf(
                    $this->formRow, $required_class, $Field->getName(), $Field->getName()
                );
            }
        }

        $tpl_html_header = $tpl_html_footer = '';


        foreach ($this->fields as $field) {

            if (is_a($field, Control\Header::class) && !is_a($field, Control\SubHeader::class)) {
                $tpl_html_header = sprintf(
                    $this->formHeader, $field->getName()
                );
            } elseif (is_a($field, Button::class) || is_a($field, Hidden::class)) {

                $tpl_html_footer .= sprintf('{{%s.control}}', $field->getName());
            }
        }

        if (!empty($tpl_html_footer)) {
            $tpl_html_footer = '<div class="panel-footer text-right">' . $tpl_html_footer . '</div>';
        }

        if (!empty($this->formHtmlBootom)) {
            $html .= $this->formHtmlBootom;
        }

        $output = '<div class="panel">' .
            $tpl_html_header .
            '<div class="panel-body">' . $html . '</div>' .
            $tpl_html_footer .
            '</div>';

        return $this->createForm(
            $output
        );
    }

    protected function isRequeredField(string $name)
    {
        if (!isset($this->validation_rules[$name]))
            return false;

        if (is_array($this->validation_rules[$name])) {

            return in_array("required", $this->validation_rules[$name]);
        }

        return strpos($this->validation_rules[$name], "required") !== false;
    }

    /**
     * @param $message_blank
     * @return $this
     * @deprecated
     * @see Simpleform::setMessageBlank()
     */
    public function setMessage_blank($message_blank)
    {
        return $this->setMessageBlank($message_blank);
    }

    /**
     * @param string $blank
     * @return Simpleform
     */
    public function setMessageBlank(string $blank): self
    {
        $this->message_blank = $blank;
        return $this;
    }

    public function createMessage($text, $type = self::MESSAGE_ERROR)
    {
        $html = sprintf($this->message_blank, $type, $text);
        //$html = preg_replace('~<a.*?href~', '<a class="alert-link" href', $html);
        //зачем менять??
        return $html;
    }

    public function addCustomMessage($message, $type = self::MESSAGE_INFO)
    {
        $this->custom_messages [] = [$message, $type];
        return $this;
    }

    public function createReturnLink()
    {
        if (!empty($this->backUrl)) {
            return sprintf(
                '<a href="%s">%s</a>', $this->backUrl, self::MESSAGE_BACK_URL
            );
        }
    }

    public function createNewLink()
    {
        if (!empty($this->newRecordUrl)) {
            return sprintf(
                '<a href="%s">%s</a>', $this->newRecordUrl, self::MESSAGE_ADD_URL
            );
        }
    }

    function setBackUrl($backUrl)
    {
        $this->backUrl = $backUrl;
        return $this;
    }

    function setNewRecordUrl($newRecordUrl)
    {
        $this->newRecordUrl = $newRecordUrl;
        return $this;
    }

    protected function getCustomMessages()
    {
        $tpl_html = '';
        if (count($this->custom_messages) > 0) {
            foreach ($this->custom_messages as $message) {
                $tpl_html .= $this->createMessage($message[0], $message[1]);
            }
        }
        return $tpl_html;
    }


    public function setClassError(string $error)
    {
        $this->classError = $error;
        return $this;
    }

    protected function getValidationMessages()
    {
        $tpl_html = '';
        if (count($this->validation_results) > 0) {
            foreach ($this->validation_results as $k => $result) {
                $tpl_html .= sprintf($this->createMessage("%s - %s", $this->classError), $this->captions[$k], $result);
            }
        }
        return $tpl_html;
    }

    /**
     * Формируем форму, средствами CI
     *
     * @param mixed $tpl
     * @return
     */
    protected function createForm($tpl)
    {

        $tpl_html = $tpl_html_header = '';

        $tpl = $this->getValidationMessages() . $this->getCustomMessages() . $tpl;

        $tpl_html .= $tpl_html_header;

        $tpl_html .= form_open_multipart($this->action,
            array(
                'class' => $this->class,
                'id' => $this->name,
                'onsubmit' => $this->onsubmit
            ), $this->hidden_fields);

        $tpl_html .= '{% autoescape false %}' . $tpl . '{% endautoescape %}';
        $tpl_html .= form_hidden('submit_' . $this->name, 1);

        $tpl_html .= form_close();

        return $tpl_html;
    }

}