<?php
/**
 * Created by PhpStorm.
 * User: Admin
 * Date: 24.08.2020
 * Time: 23:30
 */

namespace Core\Security;


use Casbin\Enforcer;
use Core\Application\Loader;
use Core\Exceptions\BaseException;
use CasbinAdapter\Database\Adapter as DatabaseAdapter;
use Diamond\Models\User\User;
use Diamond\Models\User\UserQuery;

/**
 * права доступа
 * Class Access
 * @package Core\Security
 */
class Access
{

    protected static Enforcer|null $enforcer = null;

    static private $BASE_CONF = __DIR__ . '/../../bootstrap/basic.conf';


    private static function getAdapter(): Enforcer
    {
        if (!file_exists(self::$BASE_CONF)) {
            throw new BaseException('Files rules not set');
        }

        if(static::$enforcer){
            return static::$enforcer;
        }

        $db = DatabaseAdapter::newAdapter([
            'type' => 'mysql', // mysql,pgsql,sqlite,sqlsrv
            'hostname' => getenv("DB_HOST"),
            'database' => getenv("DB_NAME"),
            'username' => getenv("DB_USER"),
            'password' => getenv("DB_PASS"),
            'hostport' => '3306',
            'policy_table_name' => '_casbin_rule'
        ]);
        static::$enforcer = new Enforcer(self::$BASE_CONF, $db);
        return static::$enforcer;
    }

    /**
     * @param string $role admin
     * @param string $object 1 or *
     * @param string $permission news_read or news_create ...
     * @return void
     */
    public function addPolicy(string $role, string $object, string $permission): void
    {
        self::getAdapter()->addPolicy($role, $object, $permission);
    }

    public function removePolicy(string $role, string $object, string $permission): void
    {
        self::getAdapter()->removePolicy($role, $object, $permission);
    }

    public function deleteAllRoleForUser(int $userId): void
    {
        self::getAdapter()->deleteRolesForUser($userId);
    }

    public function addRoleForUser(int $userId, string $role): void
    {
        self::getAdapter()->addRoleForUser((string)$userId, $role);
    }

    public function deleteRoleForUser(int $userId, string $role): void
    {
        self::getAdapter()->deleteRoleForUser((string)$userId, $role);
    }

    public function isAccess(string $domain, int $user, string $object, string $operation): bool
    {
        try {
            return self::getAdapter()->enforce((string)$user, $object, $operation);
        } catch (\TypeError $exception) {
            throw new BaseException('error access rules: ' . $exception->getMessage());
        }
    }

    public function getRulePermissions(): array
    {
        return Loader::getInstance()->load('DI')->get('permissions');
    }

    public function getRulePermissionsForRole(string $role): array
    {

        $permissions = Loader::getInstance()->load('DI')->get('permissions');
        $roles = self::getAdapter()->getImplicitPermissionsForUser($role);
        $exists = [];
        foreach ($roles as $role) {
            $key = $role[1] . '.' . $role[2];
            if (!isset($permissions[$key])) {
                continue;
            }
            $exists[$key] = $permissions[$key];
        }
        return $exists;
    }

    public function initRootRules(): void
    {
        if (!$user = \Diamond\Infrastructure\Models\User\UserQuery::create()->findOneById(1)) {
            throw new BaseException('not found user by id 1');
        }
        $role = $user->getUserGroup()->getGrpCode();
        if (!self::getAdapter()->getAllRoles()) {
            self::getAdapter()->addPolicy($role, '*', '*');
            self::getAdapter()->addRoleForUser($user->getId(), $role);
        }
    }
}