<?php

namespace Core\Console\Commands;

use Core\Exceptions\BaseException;
use Exception;
use Repo\CrudRepositoryBuilderInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use const BASEPATH;

/**
 * Description of HelloWorldCommand
 *
 * @author Dmitriy
 */
class BuildEntityCommand extends Command
{

    protected function configure()
    {
        $this
            ->setName('build:entity')
            ->setDefinition(
                new InputDefinition(array(
                    new InputOption('entity', 'e', InputOption::VALUE_REQUIRED)
                ))
            )
            ->setDescription('Create entity sceleton, build:entity -e Shop.Customer');
    }

    /**
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {

        try {
            $this->createEntity($input, $output);
        } catch (BaseException $exc) {
            $output->writeln($exc->getMessage());
        } catch (Exception $exc) {
            $output->writeln("error build entity " . $exc->getMessage());
        }
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @throws BaseException
     */
    private function createEntity(InputInterface $input, OutputInterface $output)
    {

        if (!$module = $input->getOption('entity')) {
            throw new BaseException("entity name not set");
        }

        if (!preg_match("~^(.*?)\.(.*?)$~", $module, $match)) {
            throw new BaseException("entity name incorrect");
        }

        $namespace = $match[1];

        $class = $match[2];

        $path = include_once BASEPATH . '/vendor/composer/autoload_psr4.php';

        if(isset($path[$namespace.'\\']) && isset($path[$namespace . '\\'][0])){
            $entityPathDir = $path[$namespace . '\\'][0]. '/Domain/' . $class;
        }
        elseif (isset($path[$namespace . '\Domain\\']) && isset($path[$namespace . '\Domain\\'][0])) {
            $entityPathDir = $path[$namespace . '\Domain\\'][0] . '/' . $class;
        }
        else{
            throw new BaseException(sprintf("composer psr4 not set %s path", $namespace));
        }

        //-------------------
        $namespaceFull = sprintf("%s\Domain\%s", $namespace, $class);
        $contractsPathTpl = $entityPathDir . '/Contracts';

        //-------------------
        $collectionPath = $entityPathDir . '/' . $class . "Collection.php";
        $entityInterfacePath = $contractsPathTpl . '/' . $class . "EntityInterface.php";
        $criteriaInterfacePath = $contractsPathTpl . '/' . $class . "CriteriaInterface.php";
        $repositoryInterfacePath = $contractsPathTpl . '/' . $class . "RepositoryInterface.php";


        $codeClass = <<<EOF
<?php

namespace $namespaceFull;

use Repo\Concrete\AbstractCollection;
use {$namespaceFull}\Contracts\\{$class}EntityInterface;

/**
* Description of {$class}
*
* @author Diamond Code Generator
*/
class {$class}Collection extends AbstractCollection
{

    /**
     * @return string
     */
    protected function getEntityClass(): string
    {
        return {$class}EntityInterface::class;
    }
}               
EOF;

        $entityInterfaceClass = <<<EOF
<?php

namespace $namespaceFull\Contracts;

use Repo\EntityInterface;

/**
* Description of {$class}
*
* @author Diamond Code Generator
*/
interface {$class}EntityInterface extends EntityInterface
{

}             
EOF;

        $criteriaInterfaceClass = <<<EOF
<?php

namespace $namespaceFull\Contracts;

use Repo\PaginationInterface;
use Repo\RepositoryCriteriaInterface;

/**
* Description of {$class}
*
* @author Diamond Code Generator
*/
interface {$class}CriteriaInterface extends RepositoryCriteriaInterface, PaginationInterface
{

}           
EOF;

        $repositoryInterfaceClass = <<<EOF
<?php

namespace $namespaceFull\Contracts;

use Repo\CrudRepositoryBuilderInterface;

/**
* Description of {$class}
*
* @author Diamond Code Generator
*/
interface {$class}RepositoryInterface extends CrudRepositoryBuilderInterface
{

}           
EOF;


        //BASE
        if (!is_dir($entityPathDir)) {
            mkdir($entityPathDir);
            $output->writeln("entity dir create success!");
        }

        if (!is_dir($contractsPathTpl)) {
            mkdir($contractsPathTpl);
            $output->writeln("contracts dir create success!");
        }

        if (!is_file($collectionPath)) {
            file_put_contents($collectionPath, $codeClass);
            $output->writeln("collection create success!");
        }

        if (!is_file($criteriaInterfacePath)) {
            file_put_contents($criteriaInterfacePath, $criteriaInterfaceClass);
            $output->writeln("criteria create success!");
        }

        if (!is_file($entityInterfacePath)) {
            file_put_contents($entityInterfacePath, $entityInterfaceClass);
            $output->writeln("entity interface create success!");
        }

        if (!is_file($repositoryInterfacePath)) {
            file_put_contents($repositoryInterfacePath, $repositoryInterfaceClass);
            $output->writeln("repository interface create success!");
        }
    }

}
