<?php

namespace Core\Form;

use Core\Form\Control\Captcha;
use Core\Form\Control\Input;
use Core\Form\Control\TableCompact;
use Core\Form\Control\TextArea;
use Core\Form\Control\TextEditor;
use Core\Form\Control\Uniform\Dropdown;
use Core\Form\Control\Uniform\InputTiny;
use Core\Repository\CrudInterface;
use Core\Template\Template;
use Core\TwigAwareInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use Widgets\Jquery\Tinymce\Tinymce;

/**
 * Description of TextEditorBuilder
 *
 * @author Dmitriy
 */
class ControlBuilder implements TwigAwareInterface
{

    use \Core\TwigAwareTrait;

    protected $twig;
    protected $template;
    protected $request;
    protected $session;

    /**
     * ControlBuilder constructor.
     * @param Template $template
     * @param Request $request
     * @param Session $session
     */
    function __construct(Template $template, Request $request, Session $session)
    {
        $this->template = $template;
        $this->request = $request;
        $this->session = $session;
    }

    /**
     * @param string $name
     * @param int $rows
     * @return TextArea
     */
    public function buildTextArea(string $name, $rows = 3): TextArea
    {
        return new TextArea($name, $rows);
    }

    /**
     * @param string $name
     * @param string $value
     * @return Control\Row
     */
    public function buildRow(string $name, ?string $value): Control\Row
    {
        return new Control\Row($name, $value);
    }

    /**
     * @param string $name
     * @return Input
     */
    public function buildInput(string $name): Input
    {
        return new Input($name);
    }

    /**
     * @param string $name
     * @return Control\Hidden
     */
    public function buildHidden(string $name): Control\Hidden
    {
        return new Control\Hidden($name);
    }

    /**
     * @param string $name
     * @param string $value
     * @param bool $checked
     * @return Control\Checkbox
     */
    public function buildCheckbox(string $name, string $value, bool $checked = false): Control\Checkbox
    {
        return new Control\Checkbox($name, $value, $checked);
    }

    /**
     * @param string $name
     * @param array $options
     * @param string|null $set
     * @return Control\Radio
     */
    public function buildRadio(string $name, array $options, string $set = null): Control\Radio
    {
        return new Control\Radio($name, $options, $set);
    }

    /**
     * @param string $name
     * @return Input
     */
    public function buildInputSmall(string $name): Input
    {
        return (new Input($name))->setClass("smallinput");
    }

    /**
     * @param $name
     * @return Input
     */
    public function buildInputPassword(string $name): Input
    {
        return (new Input($name))->setType("password")->setClass("smallinput");
    }

    /**
     * @param string $name
     * @return Input
     */
    public function buildInputLittle(string $name): Input
    {
        return (new Input($name))->setClass("width100");
    }

    /**
     * @param string $name
     * @param string $value
     * @return Control\Submit
     */
    public function buildSubmitButton(string $name, string $value): Control\Submit
    {
        return new Control\Submit($name, $value);
    }

    /**
     * @param string $name
     * @return InputTiny
     */
    public function buildInputTiny(string $name): InputTiny
    {
        return new InputTiny($name);
    }

    /**
     * @param string $id
     * @param string $mask
     * @return \Widgets\Jquery\Jmask\Jmask
     */
    public function buildJmask(string $id, string $mask): \Widgets\Jquery\Jmask\Jmask
    {
        return (new \Widgets\Jquery\Jmask\Jmask($this->template, $this->twig))->setId($id)->setMask($mask);
    }

    /**
     * @param string $name
     * @param array $options
     * @return Dropdown
     */
    public function buildDropdownUniform(string $name, array $options): Dropdown
    {
        return new Dropdown($name, $options);
    }

    /**
     * @param string $name
     * @param array $options
     * @return Dropdown
     */
    public function buildAutoDropdown(string $name, array $options): Dropdown
    {
        $selector = 'a-select-' . $name;
        $auto = new \Widgets\Jquery\Autocomplit\Autocomplit($selector, $this->template, $this->twig);
        $auto->Render();
        return (new Dropdown($name, $options))->setClass('chzn-select')->setId($selector);
    }

    /**
     * @param string $name
     * @param array $options
     * @return Control\DropdownMulti
     */
    public function buildAutoDropdownMulti(string $name, array $options): Control\DropdownMulti
    {
        $selector = 'a-select-' . $name;
        $auto = new \Widgets\Jquery\Autocomplit\Autocomplit($selector, $this->template, $this->twig);
        $auto->Render();
        return (new Control\DropdownMulti($name, $options))
            ->setClass('chzn-select')->setId($selector);
    }

    /**
     * @param $name
     * @return TextEditor
     */
    public function buildTextEditor(string $name): TextEditor
    {
        $editor = new Tinymce($this->template, $this->twig, $name);
        return (new TextEditor($editor, $name))->setClass('tiny_' . $name);
    }

    /**
     * @param $name
     * @return Dropdown
     */
    public function buildDropdownYNUniform(string $name): Dropdown
    {
        return new Dropdown($name, [
            "N" => "Нет",
            "Y" => "Да"
        ]);
    }

    /**
     * @param string $name
     * @return Dropdown
     */
    public function buildDropdownYNDigitUniform(string $name): Dropdown
    {
        return new Dropdown($name, [
            1 => "Да",
            0 => "Нет"
        ]);
    }

    /**
     * @param string $name
     * @param int $from
     * @param int $to
     * @return Dropdown
     */
    public function buildDropdownRankUniform(string $name, $from = 0, $to = 5)
    {
        $list = [];
        for ($i = $from; $i <= $to; $i++) {
            $list[$i] = $i;
        }

        return new Dropdown($name, $list);
    }

    /**
     * @param string $name
     * @param string $path
     * @param bool $translate
     * @return Control\ImgUpload
     */
    public function buildImgUpload(string $name, string $path, $translate = false): Control\ImgUpload
    {
        return (new Control\ImgUpload($name, $path, $this->request))->setTranslate($translate);
    }

    /**
     * @param string $name
     * @param int $width
     * @param int $height
     * @return Captcha
     */
    public function buildCaptcha(string $name, int $width = 150, int $height = 40): Captcha
    {
        return (new Captcha($this->session))->setWidth($width)->setHeight($height)->setName($name);
    }

    /**
     * @param string $name
     * @param string $key
     * @param string $secret
     * @return Control\ReCaptcha
     */
    public function buildReCaptcha(string $name, string $key, string $secret)
    {
        $this->template->addScript("<script src='https://www.google.com/recaptcha/api.js'></script>");
        return (new Control\ReCaptcha($name, $key, $secret));
    }

    /**
     * @param string $name
     * @return Control\ColorPicker
     */
    public function buildColorPicker(string $name): Control\ColorPicker
    {
        return new Control\ColorPicker($name, $this->template, $this->twig);
    }

    /**
     * @param string $name
     * @return Control\DatePicker
     */
    public function buildDatePicker(string $name): Control\DatePicker
    {
        return new Control\DatePicker($name, $this->template, $this->twig);
    }

    /**
     * @param string $name
     * @param null $value
     * @return Control\TextLabel
     */
    public function buildTextLabel(string $name, $value = null): Control\TextLabel
    {
        return new Control\TextLabel($name, $value);
    }

    /**
     * @param string $name
     * @param string $caption
     * @return Control\Button
     */
    public function buildCancel(string $name, string $caption): Control\Button
    {
        return (new Control\Button($name, $caption))
            ->setType("button")
            ->setClass("cancel asSubmit")
            ->setOnClick("location.href='" . $this->request->getpathInfo() . "'");
    }


    /**
     * @param string $name
     * @param string $caption
     * @param string $link
     * @return Control\Button
     */
    public function buildButton(string $name, string $caption, string $link = ""): Control\Button
    {
        return (new Control\Button($name, $caption))
            ->setType("button")
            ->setClass("cancel asSubmit")
            ->setOnClick($link > "" ? "location.href='" . $link . "'" : "");
    }

    /**
     * Создание компактной таблицы
     * @param string $name
     * @param array $columns
     * @return TableCompact
     */
    public function buildTableCompact(string $name, CrudInterface $repo, callable $callable,float $minPrice): TableCompact
    {
        return new TableCompact($name,$repo,$callable,$minPrice);
    }
}
