<?php

namespace Core\Form;

use Core\Form\Control\Input;
use Core\Form\Control\TextArea;
use Core\Form\Control\TextEditor;
use Core\Form\Control\Uniform\Dropdown;
use Core\Form\Control\Uniform\InputTiny;
use Core\Template\Template;
use Core\TwigAwareInterface;
use Symfony\Component\HttpFoundation\Request;
use Widgets\Jquery\Tinymce\Tinymce;
use Symfony\Component\HttpFoundation\Session\Session;
use Core\Form\Control\Captcha;

/**
 * Description of TextEditorBuilder
 *
 * @author Dmitriy
 */
class ControlBuilder implements TwigAwareInterface {

    use \Core\TwigAwareTrait;
    
    protected $twig;
    protected $template;
    protected $request;
    protected $session;

    function __construct(Template $template, Request $request, Session $session) {
        $this->template = $template;
        $this->request = $request;
        $this->session = $session;
    }

    /**
     * 
     * @param string $name
     * @param int $rows
     * @return TextArea
     */
    public function buildTextArea($name,$rows = 3) {
        return new TextArea($name,$rows);
    }

    /**
     * simple row 
     * @param type $value
     * @return \Core\Form\Control\Row
     */
    public function buildRow($name, $value): Control\Row
    {
        return new Control\Row($name,$value);
    }
    
    /**
     * 
     * @param string $name
     * @return Input
     */
    public function buildInput($name) {
        return new Input($name);
    }
    
    public function buildHidden($name) {
        return (new Control\Hidden($name));
    }
    
    public function buildCheckbox($name,$value,$checked = false) {
        return new Control\Checkbox($name,$value,$checked);
    }
    
    public function buildRadio($name,array $options, $set = null) {
        return new Control\Radio($name, $options, $set);
    }
    
    /**
     * 
     * @param string $name
     * @return Input
     */
    public function buildInputSmall($name):Input {
        return (new Input($name))->setClass("smallinput");
    }  
    
    /**
     * 
     * @param string $name
     * @return Input
     */
    public function buildInputPassword($name):Input {
        return (new Input($name))->setType("password")->setClass("smallinput");
    }    
    
    /**
     * 
     * @param string $name
     * @return Input
     */
    public function buildInputLittle($name):Input {
        return (new Input($name))->setClass("width100");
    }  
    
    /**
     * 
     * @param string $name
     * @param string $value
     * @return \Core\Form\Control\Submit
     */
    public function buildSubmitButton($name,$value):Control\Submit {
        return  new Control\Submit($name,$value);
    }
   
    /**
     * 
     * @param string $name
     * @return InputTiny
     */
    public function buildInputTiny($name):InputTiny {
        return new InputTiny($name);
    }
    
    /**
     * 
     * @param string $id
     * @param string $mask
     * @return \Widgets\Jquery\Jmask\Jmask
     */
    public function buildJmask($id,$mask): \Widgets\Jquery\Jmask\Jmask{
        return (new \Widgets\Jquery\Jmask\Jmask($this->template, $this->twig))->setId($id)->setMask($mask);
    }

    /**
     * 
     * @param type $name
     * @param array $options
     * @return Dropdown
     */
    public function buildDropdownUniform($name, array $options): Dropdown {
        return new Dropdown($name, $options);
    }
    
    public function buildAutoDropdown($name,array $options) {
        
        $selector = 'a-select-'.$name;
        $auto = new \Widgets\Jquery\Autocomplit\Autocomplit($selector,$this->template, $this->twig);
        $auto->Render();
        return (new Dropdown($name, $options))->setClass('chzn-select')->setId($selector);
    }
    
    public function buildAutoDropdownMulti($name,array $options) {
        
        $selector = 'a-select-'.$name;
        $auto = new \Widgets\Jquery\Autocomplit\Autocomplit($selector,$this->template, $this->twig);
        $auto->Render();
        return (new Control\DropdownMulti($name, $options))
                
                ->setClass('chzn-select')->setId($selector);
    }
    
    /**
     * 
     * @param type $name
     * @return TextEditor
     */
    public function buildTextEditor($name):TextEditor{
        $editor = new Tinymce($this->template, $this->twig, $name);
        return (new TextEditor($editor, $name))->setClass('tiny_' . $name);
    }    
    
    /**
     * 
     * @param type $name
     * @return Dropdown
     */
    public function buildDropdownYNUniform($name):Dropdown{
        return new Dropdown($name, [
            "N" =>  "Нет",
            "Y" =>  "Да"
        ]);
    }
    
    /**
     * 
     * @param type $name
     * @param type $from
     * @param type $to
     * @return Dropdown
     */
    public function buildDropdownRankUniform($name,$from=0,$to=5)
    {
        $list = [];
        for($i=$from;$i<=$to;$i++){
            $list[$i]=$i;
        }
        
        return new Dropdown($name,$list);
    }

    /**
     * 
     * @param string $name
     * @param string $path
     * @param boolean $translate
     * @return type
     */
    public function buildImgUpload($name, $path, $translate = false):Control\ImgUpload {
        return (new Control\ImgUpload($name, $path, $this->request))->setTranslate($translate);
    }

    /**
     * 
     * @param type $width
     * @param type $height
     * @return \Core\Form\Control\Captcha
     */
    public function buildCaptcha($name, $width = 150, $height = 40) {
        return (new Captcha($this->session))->setWidth($width)->setHeight($height)->setName($name);
    }
    
    /**
     * 
     * @param type $name
     */
    public function buildReCaptcha($name,$key,$secret) {
        $this->template->addScript("<script src='https://www.google.com/recaptcha/api.js'></script>");
        return (new Control\ReCaptcha($name,$key,$secret));
    }    
    /**
     * 
     * @param type $name
     * @return \Core\Form\Control\ColorPicker
     */
    public function buildColorPicker($name) {
        return new Control\ColorPicker($name,$this->template, $this->twig);	
    }
    
    /**
     * 
     * @param type $name
     * @return \Core\Form\Control\DatePicker
     */
    public function buildDatePicker($name) {
        return new Control\DatePicker($name,$this->template, $this->twig);
    }
    
    /**
     * Текстовая надпись
     * @param type $name
     * @return \Core\Form\Control\TextLabel
     */
    public function buildTextLabel($name,$value = null) {
        return new Control\TextLabel($name,$value);
    }

    /**
     * Возврат в спрачоник
     * @param type $name
     * @param type $caption
     * @return type
     */
    public function buildCancel($name,$caption) {
       
        return (new Control\Button($name,$caption))
                ->setType("button")
                ->setClass("cancel asSubmit")
                ->setOnClick("location.href='" . $this->request->getpathInfo() . "'");
    }
    
    
    /**
     * Возврат в спрачоник
     * @param type $name
     * @param type $caption
     * @return type
     */
    public function buildButton($name,$caption,$link = "") {
       
        return (new Control\Button($name,$caption))
                ->setType("button")
                ->setClass("cancel asSubmit")
                ->setOnClick($link > ""?"location.href='" . $link . "'":"");
    }    
}
